/*********************************************************************
*                    SEGGER Microcontroller GmbH                     *
*                        The Embedded Experts                        *
**********************************************************************
*                                                                    *
*            (c) 2014 - 2019 SEGGER Microcontroller GmbH             *
*                                                                    *
*       www.segger.com     Support: support@segger.com               *
*                                                                    *
**********************************************************************
*                                                                    *
* All rights reserved.                                               *
*                                                                    *
* Redistribution and use in source and binary forms, with or         *
* without modification, are permitted provided that the following    *
* condition is met:                                                  *
*                                                                    *
* o Redistributions of source code must retain the above copyright   *
*   notice, this condition and the following disclaimer.             *
*                                                                    *
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND             *
* CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,        *
* INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF           *
* MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE           *
* DISCLAIMED. IN NO EVENT SHALL SEGGER Microcontroller BE LIABLE FOR *
* ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR           *
* CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT  *
* OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;    *
* OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF      *
* LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT          *
* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE  *
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH   *
* DAMAGE.                                                            *
*                                                                    *
**********************************************************************

-------------------------- END-OF-HEADER -----------------------------

File    : SVCHandler_ARM.S
Purpose : Supervisor Call exception handler for ARM, Cortex-A and 
          Cortex-R processors to handle semihosting SVCs when no 
          debugger is connected.
*/

        .syntax unified


/*********************************************************************
*
*       Extern symbols
*
**********************************************************************
*/

        .extern SEGGER_SEMIHOST_Call
        
/*********************************************************************
*
*       Global functions
*
**********************************************************************
*/

/*********************************************************************
*
*      SVC_Handler()
*
*  Function description
*    SVC handler implementation to return on semihosting SVCs.
*
*  Notes
*    (1) Ensure that SVC_Handler is part of the exception table
*    
*/
        .section .text.SVC_Handler
        .global SVC_Handler
        .type   SVC_Handler, function
        .balign 4
#if __thumb__
        .thumb
        .thumb_func
#else
        .arm
#endif

#if _USE_SIMPLE_SVC != 0
SVC_Handler:
        //
        // SVC is used for semihosting only.
        // Simply return from SVC Handler.
        //
        subs PC, LR, 0
        .end
#else
SVC_Handler:
        push {R2,R3}
        //
        // Check whether we come from Thumb or ARM mode.
        // Application's CPSR is saved in SPSR.
        //
        mrs R2, SPSR
        tst R2, #0x20
        beq _CheckSemiARM               // SPSR.T (Bit 5) not set -> ARM mode
_CheckSemiThumb:
        //
        // Load immediate of Thumb SVC instruction
        //
#if BIG_ENDIAN
        ldrb R2, [LR, #-2]
#else
        ldrb R2, [LR, #-1]
#endif
        //
        // Check if it has been a semihosting SVC
        //
        ldrb R3, =0xAB
        cmp R2, R3
        pop {R2,R3}                     // Restore registers (might be used by semihosting or SVC handler)
        bne _DoSVC                      // No semihosting SVC. Continue in SVC handler
        b _DoSemihost                   // Semihosting SVC. Run through debugger breakpoint and return
_CheckSemiARM:
        //
        // Load immediate of ARM SVC instruction
        //
        ldr R2, [LR, #-4]
        bic R2, R2, #0xFF000000
        //
        // Check if it has been a semihosting SVC
        //
        ldr R3, =0x00123456
        cmp R2, R3
        //
        // Restore registers (might be used by semihosting or SVC handler)
        //
        pop {R2,R3}
        bne _DoSVC                      // No semihosting SVC. Continue in SVC handler
        //b _DoSemihost                   // Semihosting SVC. Run through debugger breakpoint and return
_DoSemihost:
        //
        // Call semihosting handler for the debugger to set a breakpoint on
        //
        blx SEGGER_SEMIHOST_DebugHalt
        //
        // Perform exception return to instruction after SVC.
        // Also resets CPSR.
        //
        subs PC, LR, 0
_DoSVC:
        //
        // Stay in endless loop
        // User SVC handler could start here instead
        //
        b _DoSVC
        .end
#endif

/*************************** End of file ****************************/
